"use client"

import LoadingCircle from "@/components/icons/loading.circle";
import { Button } from "@/components/ui/button";
import { DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toaster";
import { unknownErrorToStringFormatter } from "@/lib/formatters/error-formatters";
import { executeOrReject, sleep } from "@/lib/timing";
import { TransformedAgency } from "@/types/transformers";
import { ComponentWithClassName } from "@/types/utils";
import { zodResolver } from "@hookform/resolvers/zod";
import { uploadData } from "aws-amplify/storage";
import React, { ComponentProps } from "react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { generateFileUploadPath, triggerTripPlannerBuild } from "../lib/actions";
import { InputFileUpload } from "./input.file-upload";

const INTERNAL__MAX_SIZE = (16 * 1024 * 1024) satisfies ComponentProps<typeof InputFileUpload>["maxSize"]
const INTERNAL__MAX_COUNT = (15) as const satisfies ComponentProps<typeof InputFileUpload>["maxFileCount"]
const INTERNAL__ACCEPT_FILE = {
  "text/csv": [".csv"],
  "application/json": [".json", ".geojson"],
  "text/plain": [".txt"],
  "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": [".xlsx"],
  "application/vnd.ms-excel": [".xls"],
} as const satisfies ComponentProps<typeof InputFileUpload>["accept"]

const schema = z.object({
  files: z.array(z.instanceof(File)).min(1),
})

type Schema = z.infer<typeof schema>

export function WorkflowDialogUpload({ data, className }: ComponentWithClassName<{ data: NonNullable<TransformedAgency> }>) {
  const { toast } = useToast()
  const [progresses, setProgresses] = React.useState<Record<string, number>>({})
  const isUploading = false;
  const form = useForm<Schema>({
    resolver: zodResolver(schema),
    defaultValues: {
      files: [],
    },
  })

  if (data.access !== "ALL") {
    return;
  }
  const { isLoading, isValid } = form.formState
  const value = form.watch()

  async function onSubmit(input: Schema) {
    if (input.files.length === 0) {
      toast({
        duration: 2000,
        title: "No files to upload",
        description: "Click on or drag files to the file drop input to upload them to the server.",
        variant: "info",
      })

      return;
    }

    const uploadingFilesToast = toast({
      duration: Infinity,
      title: "Uploading files",
      variant: "loading",
      description: "Do not navigate away from this page. Changes may corrupt uploaded files.",
      disableDismiss: true,
      onDismissed() {
        setProgresses({});
      }
    })

    try {
      const handlers = await Promise.all(input.files.map(async (f) => {
        const name = f.name
        const path = await generateFileUploadPath(data.path, name);
        return uploadData({
          data: f,
          path: path,
          options: {
            onProgress(e) {
              const percentage = (e.transferredBytes / (e.totalBytes ?? 1)) * 100
              setProgresses((p) => {
                p[name] = percentage
                return p
              })
            },
          }
        })
      }))

      await Promise.all(handlers.map((h) => {
        return h.result
      }))

      const potentialErrors = handlers.filter((h) => h.state === 'ERROR')
      if (potentialErrors.length > 0) {
        const filesWithErrors = await Promise.all(potentialErrors.map(async (e) => {
          const file = await e.result
          return file.path
        }))

        const message = `One or more files failed to upload:\n\n${filesWithErrors.join(', ')}`;
        uploadingFilesToast.dismiss()
        setProgresses({});
        toast({
          duration: Infinity,
          title: "Error uploading files",
          variant: "error",
          description: message,
          action: {
            label: 'Copy error',
            altText: 'Copy error',
            onClick: () => {
              navigator.clipboard.writeText(message)
            }
          }
        })

        return;
      }

      form.reset();
      setProgresses({});
      const token = { cancelled: false }
      uploadingFilesToast.update({
        title: "Triggering new build.",
        description: "A build will begin in 5 seconds. Cancelling the build may affect preview trip planner.",
        variant: "loading",
        action: {
          label: 'Cancel build',
          altText: 'Cancel build',
          onClick: () => {
            token.cancelled = true
          }
        }
      })

      await executeOrReject(() => triggerTripPlannerBuild(data.gtfsId), 5000, token)
      uploadingFilesToast.dismiss()

      await sleep(500)
      if (token.cancelled) {
        toast({
          duration: 3000,
          title: "No new build.",
          description: "Upload files to trigger a new build.",
          variant: "warning",
        })
      } else {
        toast({
          title: "Files uploaded successfully!",
          variant: "success",
          description: "All files were uploaded and a new trip planner server build is currently running. Check back later to see progress",
        })
      }
    } catch (error) {
      uploadingFilesToast.dismiss()
      setProgresses({});

      const message = `Errors occured during upload ${unknownErrorToStringFormatter(error)}`;
      toast({
        duration: Infinity,
        title: "Error during upload",
        variant: "error",
        description: message,
        action: {
          label: 'Copy error',
          altText: 'Copy error',
          onClick: () => {
            navigator.clipboard.writeText(message)
          }
        }
      })
    }
  }

  return (
    <>
      <DialogHeader className={className}>
        <DialogTitle>Upload new feed</DialogTitle>
        <DialogDescription>Currently uploading files for &apos;<span className="font-mono font-bold">{data.name}</span>&apos;.</DialogDescription>
      </DialogHeader>

      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(onSubmit)}
          className="flex-1 flex w-full flex-col gap-6"
        >
          <FormField
            control={form.control}
            name="files"
            render={({ field }) => (
              <div className="space-y-6">
                <FormItem className="w-full">
                  <FormControl>
                    <InputFileUpload
                      value={field.value}
                      onValueChange={field.onChange}
                      accept={INTERNAL__ACCEPT_FILE}
                      maxFileCount={INTERNAL__MAX_COUNT}
                      maxSize={INTERNAL__MAX_SIZE}
                      progresses={progresses}
                      disabled={isUploading}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              </div>
            )}
          />

          <div className="w-full h-full flex-1 grid place-items-end">
            <div className="flex flex-row gap-1">
              <Button type="reset" variant={"secondary"} className="w-fit flex flex-row gap-2" disabled={isLoading || !isValid} onClick={() => form.reset()}>
                Reset files
              </Button>

              <Button type="submit" className="w-fit flex flex-row gap-2" disabled={isLoading || !isValid}>
                Upload {value.files.length} file(s) and start build
                {isLoading && <LoadingCircle className="size-3" />}
              </Button>
            </div>
          </div>
        </form>
      </Form>
    </>
  )
}

